/** @file   bullettable.cpp
 * @brief   Implementation of BulletTable - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "bullettable.h"
#include "animplayer.h"
#include "gameanims.h"
#include "gameobject.h"
#include "www_assert.h"
#include "soundsamples.h"
#include "AlienFireballBullet.h"
#include "FlameThrowerBullet.h"
#include "RocketBullet.h"
#include "SentryFireballBullet.h"
#include "TankBullet.h"
#include "MinigunBullet.h"
using namespace eng2d;
using std::list;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================

/** Creates new bullets
 */
Bullet* BulletTable::createBullet( GameObject* aShooter, const Vec2D& aPos,
                                   Bullet::TType aType )
{
  Bullet* bullet = 0;
  switch ( aType ) {
    case ( Bullet::EAlienFireball ): {
      bullet = new AlienFireballBullet( aShooter, aPos );
      break;
    }
    case ( Bullet::EAlienFireball2 ): {
      bullet = new AlienFireballBullet( aShooter, aPos );
      bullet->iType = Bullet::EAlienFireball2;
      break;
    }
    case ( Bullet::EFlameThrower ): {
      bullet = new FlameThrowerBullet( aShooter, aPos );
      break;
    }
    case ( Bullet::ETankBullet ): {
      bullet = new TankBullet( aShooter, aPos );
      break;
    }
    case ( Bullet::ERocketLauncher ): {
      bullet = new RocketBullet( aShooter, aPos );
      break;
    }
    case ( Bullet::ESentryFireball ): {
      bullet = new SentryFireballBullet( aShooter, aPos );
      break;
    }
    case ( Bullet::EMinigun ): {
      bullet = new MinigunBullet( aShooter, aPos );
      break;
    }
    default: {
      bullet = new Bullet( aShooter, aType, aPos );
      break;
    }
  }
  return bullet;
}



/** Creates new grenade
 */
Grenade* BulletTable::createGrenade( GameObject* aThrownBy,
                                     const Vec2D& aPos,
                                     Grenade::TType aType )
{
  Weapon weapon( Weapon::W_GRENADE );
  Grenade* grenade = new Grenade( aThrownBy );

  // Set the members of Grenade - class.
  grenade->iType = aType;
  grenade->iLifetime = weapon.reload();
  grenade->iFragmentCount = weapon.deviation();
  grenade->iDamage = weapon.damage();

  // Set the members of Ammunition - class.
  grenade->iPosition = aPos;
  grenade->iAltitude = 50;
  grenade->iFlags = Ammunition::KBounce;
  
  Vec2D mvec( 0, weapon.velocity() );
  if ( aThrownBy != 0 ) {
    mvec.rotate( aThrownBy->angle() );
  }
  grenade->setVelocity( mvec );

  return grenade;
}

///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
BulletTable::BulletTable()
{
}



/** Destructor
 */
BulletTable::~BulletTable()
{
  this->cleanup();
}




///
/// Public methods
/// ==============

/** Adds the given bullet
 */
void BulletTable::spawnBullet( Bullet* pB )
{
  iBullets.push_back( pB );
}



/** Adds the given grenade
 */
void BulletTable::spawnGrenade( Grenade* pG )
{
  iGrenades.push_back( pG );
}




/** Destroys the bullets and grenades
 */
void BulletTable::cleanup()
{
  list<Bullet*>::iterator it = iBullets.begin();
  while ( it != iBullets.end() ) {
    delete ( *it );
    it++;
  }
  iBullets.clear();
}



/** Updates the bullets
 */
void BulletTable::update()
{
  list<Bullet*>::iterator bulletIter = iBullets.begin();
  while ( bulletIter != iBullets.end() ) {
    Bullet* pB = *bulletIter;
    pB->update();
    
    // Check if the bullet has the "done"-flag on.
    if ( pB->iDone == true ) {
      // The bullet should be deleted.
      bulletIter = this->deleteBullet( bulletIter );
      
    } else {
      bulletIter++;
      
    }
  }
  
  // Update the grenades
  bool explodingGrenades = false;
  list<Grenade*>::iterator grenadeIter = iGrenades.begin();
  while ( grenadeIter != iGrenades.end() ) {
    Grenade* pG = *grenadeIter;
    if ( pG->update() == true ) {
      // We should explode this grenade
      pG->endingActions( this );
      //this->explodeGrenade( pG );
      explodingGrenades = true;
      grenadeIter = iGrenades.erase( grenadeIter );
      
    } else {
      grenadeIter++;
      
    }
  }
  
  if ( explodingGrenades == true ) {
    Sound::playSample( SMP_GRENADE, false );
  }
  
}



/** Draws the bullets and grenades
 */
void BulletTable::redraw( BITMAP* aTarget )
{
  list<Bullet*>::iterator bulletIter = iBullets.begin();
  while ( bulletIter != iBullets.end() ) {
    (*bulletIter)->redraw( aTarget );
    bulletIter++;
  }
  
  list<Grenade*>::iterator grenadeIter = iGrenades.begin();
  while ( grenadeIter != iGrenades.end() ) {
    (*grenadeIter)->redraw( aTarget );
    grenadeIter++;
  }
}



/** Detonates the grenades thrown by given GameObject
 */
void BulletTable::detonateGrenades( GameObject* aOwner )
{
  GrenadeIter grenIT = this->grenadeBegin();
  bool grenadeDetonated = false;
  while ( grenIT != this->grenadeEnd() ) {
    if ( (*grenIT)->detonate( aOwner ) == true ) {
      grenadeDetonated = true;
    }
    grenIT++;
  }
  
  if ( grenadeDetonated == true ) {
    Sound::playSample( SMP_DETONATE, false );
  }
}



/** Deletes the bullet pointed by given iterator
 */
BulletTable::BulletIter BulletTable::deleteBullet( BulletIter it )
{
  if ( it == this->bulletEnd() ) {
    return it;
  }
  (*it)->endingActions( this );
  delete (*it);
  return iBullets.erase( it );
}



///
/// Getter methods
/// ==============

/** Returns the number of bullets there are
 */
int BulletTable::bulletCount() const
{
  return iBullets.size();
}



/** Returns the number of grenades
 */
int BulletTable::grenadeCount() const
{
  return iGrenades.size();
}



/** Returns an iterator to the first bullet in the bullettable
 */
BulletTable::BulletIter BulletTable::bulletBegin()
{
  return iBullets.begin();
}



/** Returns an iterator to the first grenade in the bullettable
 */
BulletTable::GrenadeIter BulletTable::grenadeBegin()
{
  return iGrenades.begin();
}



/** Returns an iterator to the last bullet in the bullettable
 */
BulletTable::BulletIter BulletTable::bulletEnd()
{
  return iBullets.end();
}



/** Returns an iterator to the last grenade in the bullettable
 */
BulletTable::GrenadeIter BulletTable::grenadeEnd()
{
  return iGrenades.end();
}




///
/// Private or Protected methods
/// ============================


} // end of namespace
